/* FILE: mdemux.c                               (D. Tottingham  03/24/91)

This is a collection of C functions that manage the demux data queue for
xdetect.  All functions have been written and compiled medium model. The
following functions are included:

dm_get_current_time ()          get current time
dm_get_first_buffer ()          get first new demux buffer from demux queue
dm_get_head_buffer ()           get buffer at head of demux queue
dm_get_new_buffers ()           get buffer(s) from ADC and demultiplex
dm_get_next_buffer ()           get next buffer from demux queue
dm_get_pre_event_time ()        get pre event time
dm_get_tail_buffer ()           get buffer at tail of demux queue
dm_get_uptime ()                get session up time.
dm_initialize ()                initialize the demux queues
dm_initialize_params ()         initialize parameters
dm_initialize_time ()           initialize the time
dm_reset ()                     reset this module and release all storage
dm_set_PreEventTime ()          set PreEventTime constant

EXTERNAL FUNCTIONS CALLED:

dt_get_channel_size ()          get channel block size
dt_get_digitization_rate ()     get digitization rate
dt_get_new_buffers ()           get new mux buffer(s) from DT2821
dt_get_next_buffer ()           get buffer at tail of mux queue
dt_get_number_of_ext_buffers () get number of external buffers
dt_get_scan_count ()            get scan count
er_abort ()                     display an error message then quit
o_write_logfile ()              write string to log file stream
q_dequeue ()                    dequeue a data link from a data queue
q_enqueue ()                    enqueue a data link on a data queue
q_initialize ()                 initialize a data queue
suds_initialize ()              initialize a suds structure
suds_initialize_tag ()          initialize a suds structtag
u_convert_time ()               convert a struct timeb into abs. time

HISTORY:
   none

*/


/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <malloc.h>
#include <stdio.h>
#include <sys\types.h>
#include <sys\timeb.h>
#include <time.h>

#include "mconst.h"
#include "mdemux.h"
#include "mdt28xx.h"
#include "merror.h"
#include "mlog.h"
#include "mqueue.h"
#include "msudsini.h"
#include "mutils.h"
#include "powrstor.h"
#include "xdetect.h"


/*************************************************************************
                                GLOBALS

*************************************************************************/
PRIVATE Q_QUEUE demux_queue;
PRIVATE Q_LINK * head_ptr, * first_ptr;
PRIVATE double current_time, startup_time;
PRIVATE double pre_event_time;


/*=======================================================================*
 *                             adjust_queue                              *
 *=======================================================================*/
/* Adjust the pre-trigger queue and return the used ST_heap_ptr.         */

PRIVATE
struct STORheap far * adjust_queue (buffer_time)
double buffer_time;
{
   struct STORheap far * sheap_ptr;
   STORstatus stat;
   Q_TYPE type;

   sheap_ptr = NULL;
   if ((current_time - startup_time) > (pre_event_time + buffer_time)) {
      if (q_dequeue (&demux_queue, &type)) {
         sheap_ptr = type.buffer->ST_heap_ptr;
         _ffree (type.buffer);
      }
   }
   return (sheap_ptr);
}

/*=======================================================================*
 *                      create_available_buffers                         *
 *=======================================================================*/
/* Convert NEW buffers into AVAILABLE buffers.                           */

PRIVATE
void create_available_buffers ()
{
   Q_LINK * hptr;

   for (hptr = demux_queue.head; hptr != NULL; hptr = hptr->next)
      if (hptr->type.buffer->buffer_status == NEW_BUFFER) {
         hptr->type.buffer->buffer_status = AVAILABLE_BUFFER;
      }
}

/*=======================================================================*
 *                        get_available_buffer                           *
 *=======================================================================*/
/* Find an available data buffer.                                        */

PRIVATE
int far * get_available_buffer ()
{
   Q_LINK * hptr;

   for (hptr = demux_queue.head;
        hptr != NULL && hptr->type.buffer->buffer_status != NEW_BUFFER;
        hptr = hptr->next)
      if (hptr->type.buffer->buffer_status == AVAILABLE_BUFFER) {
         hptr->type.buffer->buffer_status = ARCHIVED_BUFFER;
         return (hptr->type.buffer->data);
      }
   return (NULL);
}

/*=======================================================================*
 *                               get_data                                *
 *=======================================================================*/
/* If data is not in base memory, get it from the Sheap.                 */

PRIVATE
void get_data (buffer)
Q_BUFFER far * buffer;
{
   STORstatus stat;
   unsigned int actual_buffer_size;

   if (buffer == NULL || buffer->buffer_status != ARCHIVED_BUFFER)
      return;

   actual_buffer_size = buffer->buffer_size * sizeof(unsigned);
   if ((buffer->data = get_available_buffer()) == NULL) {
      buffer->data = (int far *) _fmalloc ( actual_buffer_size);
      if (buffer->data == NULL) er_abort (DM_NO_STORAGE);
   }

   buffer->buffer_status = AVAILABLE_BUFFER;
   stat = getSTOR(*(buffer->ST_heap_ptr), 0L, buffer->data, actual_buffer_size);
   if (stat != Okay) er_abort (DM_POWRSTOR + stat);
}

/*=======================================================================*
 *                         dm_get_current_time                           *
 *=======================================================================*/
/* Get current time.                                                     */

PUBLIC
double dm_get_current_time ()
{
   return (current_time);
}

/*=======================================================================*
 *                         dm_get_first_buffer                           *
 *=======================================================================*/
/* Get first new demux buffer from demux queue.                          */

PUBLIC
Q_BUFFER far * dm_get_first_buffer ()
{
   head_ptr = first_ptr;
   if (head_ptr != NULL) {
      get_data (head_ptr->type.buffer);
      return (head_ptr->type.buffer);
   } else return (NULL);
}

/*=======================================================================*
 *                          dm_get_head_buffer                           *
 *=======================================================================*/
/* Get buffer at head of queue.                                          */

PUBLIC
Q_BUFFER far * dm_get_head_buffer ()
{
   head_ptr = demux_queue.head;
   if (head_ptr != NULL) {
      get_data (head_ptr->type.buffer);
      return (head_ptr->type.buffer);
   } else return (NULL);
}

/*=======================================================================*
 *                          dm_get_new_buffers                           *
 *=======================================================================*/
/* Get buffer from ADC.  Update buffer_start_time.  Demultiplex the raw
   data and save link in data queue.                                     */

PUBLIC
void dm_get_new_buffers ()
{
   Q_BUFFER far * mux, far * demux;
   Q_TYPE type;
   STORstatus stat;
   unsigned int actual_buffer_size;

   /* Wait for a buffer to complete */
   dt_get_new_buffers ();
   first_ptr = NULL;

   /* Free up the new demux data buffers */
   create_available_buffers();

   while ( (mux = dt_get_next_buffer()) != NULL) {
      demux = (Q_BUFFER far *) _fmalloc (sizeof (Q_BUFFER));
      if (demux == NULL) er_abort (DM_NO_STORAGE);

      suds_initialize (MUXDATA, &demux->info);
      suds_initialize_tag (MUXDATA, &demux->structtag);

      demux->bank_switched = mux->bank_switched;
      demux->dynamic_range = mux->dynamic_range;
      demux->buffer_status = NEW_BUFFER;
      demux->buffer_size = mux->buffer_size;
      demux->structtag = mux->structtag;
      demux->info = mux->info;
      demux->info.begintime = current_time;
      demux->info.loctime = 0;

      actual_buffer_size = mux->buffer_size * sizeof(unsigned);
      demux->data = get_available_buffer();
      if (demux->data == NULL) {
         demux->data = (int far *) _fmalloc ( actual_buffer_size);
         if (demux->data == NULL) er_abort (DM_NO_STORAGE);
      }

      demux->ST_heap_ptr = adjust_queue (mux->seconds_in_buffer);
      if (demux->ST_heap_ptr == NULL) {
         demux->ST_heap_ptr = (struct STORheap far *) _fmalloc ( sizeof(struct STORheap));
         if (demux->ST_heap_ptr == NULL) er_abort (DM_NO_STORAGE);
         stat = newSheap( demux->ST_heap_ptr);
         if (stat != Okay) er_abort (DM_POWRSTOR + stat);
         stat = setSheap( *(demux->ST_heap_ptr),
                          (unsigned long) (actual_buffer_size));
         if (stat != Okay) er_abort (DM_POWRSTOR + stat);
      }

      /* Demultiplex the mux buffer */
      to_demux (mux->data, demux->data, mux->buffer_size *
                sizeof(unsigned), mux->info.numchans);

      /* Copy the demultiplexed data into the Sheap */
      stat = putSTOR( *(demux->ST_heap_ptr), demux->data, 0L, actual_buffer_size);
      if (stat != Okay) er_abort (DM_POWRSTOR + stat);

      /* Put the demux buffer in the queue */
      type.buffer = demux;
      q_enqueue (&demux_queue, type);

      current_time += mux->seconds_in_buffer;
      if (first_ptr == NULL) first_ptr = demux_queue.tail;
   }
}

/*=======================================================================*
 *                         dm_get_next_buffer                            *
 *=======================================================================*/
/* Get next buffer from queue.                                           */

PUBLIC
Q_BUFFER far * dm_get_next_buffer ()
{
   head_ptr = head_ptr->next;
   if (head_ptr != NULL) {
      get_data (head_ptr->type.buffer);
      return (head_ptr->type.buffer);
   } else return (NULL);
}

/*=======================================================================*
 *                        dm_get_pre_event_time                          *
 *=======================================================================*/
/* Get pre event time.                                                   */

PUBLIC
double dm_get_pre_event_time ()
{
   return (pre_event_time);
}

/*=======================================================================*
 *                         dm_get_tail_buffer                            *
 *=======================================================================*/
/* Get buffer at tail of queue.                                          */

PUBLIC
Q_BUFFER far * dm_get_tail_buffer ()
{
   if (demux_queue.tail != NULL) {
      get_data (demux_queue.tail->type.buffer);
      return (demux_queue.tail->type.buffer);
   } else return (NULL);
}

/*=======================================================================*
 *                            dm_get_uptime                              *
 *=======================================================================*/
/* Get session up time.                                                  */

PUBLIC
double dm_get_uptime ()
{
   return (current_time - startup_time);
}

/*=======================================================================*
 *                             dm_initialize                             *
 *=======================================================================*/
/* Initialize the queues and calculate some key values.                  */

PUBLIC
void dm_initialize ()
{
   static union INITelement init_storage[2];
   STORstatus stat;
   int low_limit, high_limit;
   unsigned int needed_size, chunks_in_block, blocks_in_queue;
   unsigned long samples_in_queue, bytes_in_block, channel_size;

   /* Dedicate an area of extended memory to PowerSTOR. Upper and
      lower limits are specified in 16K chunks from the bottom of
      base memory.  Note that there are 8 16K chunks in each ATLAB
      buffer (128K bytes) and 64 16K chunks to the 1M byte line.
    */
   low_limit = 64;
   high_limit = -(dt_get_number_of_ext_buffers() * 8 + 1);

   /* Calculate the amount of PowerSTOR needed to hold the pre-trigger queue */
   channel_size = dt_get_channel_size();
   bytes_in_block = channel_size * dt_get_scan_count() * sizeof(unsigned);
   chunks_in_block = bytes_in_block / SMALLEST_CHUNK;
   if (bytes_in_block % SMALLEST_CHUNK) chunks_in_block++;
   samples_in_queue = pre_event_time * dt_get_digitization_rate();
   blocks_in_queue = (samples_in_queue / channel_size) + 1;
   if (samples_in_queue % channel_size) blocks_in_queue++;
   needed_size = chunks_in_block * blocks_in_queue + POWRSTOR_USAGE;
   if (needed_size < 4) needed_size = 4;

/** debug **/
if (Debug_enabled) {
   printf ("PowerSTOR stats: \n");
   printf (" low_limit, high_limit = %d, %d\n", low_limit, high_limit);
   printf (" chunks_in_block, blocks_in_queue, needed_size = %u, %u, %u\n",
           chunks_in_block, blocks_in_queue, needed_size);
}

   /* Fill the init_storage array */
/**/
   init_storage[0].EXTmem.memType = ExtMem;
   init_storage[0].EXTmem.LoLimit = low_limit;
   init_storage[0].EXTmem.HiLimit = high_limit;
/**/
   init_storage[1].Nulmem.memType = NulMem;
/**
   init_storage[0].diskF.memType = DskFile;
   init_storage[0].diskF.MaxSize = needed_size;
   strcpy(init_storage[0].diskF.namestring, "d:file.tmp");
**/

   /* Initialize PowerSTOR */
   stat = InitPwr( needed_size, init_storage);
   if (stat != Okay) er_abort (DM_POWRSTOR + stat);

   q_initialize (&demux_queue);
   first_ptr = head_ptr = NULL;
}

/*=======================================================================*
 *                          dm_initialize_params                         *
 *=======================================================================*/
/* Initialize parameters.                                                */

PUBLIC
void dm_initialize_params ()
{
   pre_event_time = PRE_EVENT_TIME;
}

/*=======================================================================*
 *                           dm_initialize_time                          *
 *=======================================================================*/
/* Initialize the time.                                                  */

PUBLIC
void dm_initialize_time ()
{
   struct timeb current_timeb;

   ftime (&(current_timeb));
   startup_time = current_time = u_convert_time (current_timeb);
}

/*=======================================================================*
 *                                dm_reset                               *
 *=======================================================================*/
/* Reset this module and release all storage.                            */

PUBLIC
void dm_reset ()
{
   Q_TYPE type;

   while (q_dequeue (&demux_queue, &type)) {
      if (type.buffer->buffer_status != ARCHIVED_BUFFER)
         _ffree (type.buffer->data);
      _ffree (type.buffer->ST_heap_ptr);
      _ffree (type.buffer);
   }
   QuitPwr();
}

/*=======================================================================*
 *                           dm_set_PreEventTime                         *
 *=======================================================================*/
/* Set PreEventTime constant.                                            */

PUBLIC
void dm_set_PreEventTime (PreEventTime)
double PreEventTime;
{
   pre_event_time = PreEventTime;
}
